/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: $
   Last Modification:
    $Author: stephans $
    $Modtime: $
    $Revision: 6511 $

   Targets:
     MQX          : yes

   Description:
    MQX OS Abstraction Implementation

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         01.03.2011  SS       initial version

**************************************************************************************/

/* Standard includes */
#include "OS_Includes.h"
#include <string.h>
#include <ctype.h>

/* MQX includes */
#include <mqx.h>
#include <mutex.h>
#include <bsp.h>
#include <fio.h>
#include <io_gpio.h>

/* Toolkit includes */
#include "cifXToolkit.h"
#include "OS_Dependent.h"
#include "cifXErrors.h"
#include "MQXSerDPMInterface.h"
#include "MQXParDPMInterface.h"

/* Serial DPM includes */
#include "OS_Spi.h"

static const VMCF51CN_STRUCT_PTR s_ptMCF51CN   = (VMCF51CN_STRUCT_PTR)0xFFFF8000;
static       void*               s_pvInterrupt = NULL;
static void*                     s_pvChipSelect = NULL;
static       LWSEM_STRUCT        s_tIsrSem     = {0};

#if (NX50_PARDPM8 == 0) & (NX50_PARDPM16 == 0) & (NX50_SERDPM == 0)
#error "One interface must be selected"
#endif

/*****************************************************************************/
/*! Workaround for standard BSP 
 *  Overwrite sci init struct to support sci0 instead of sci1 for i/o        */ 
/*****************************************************************************/
const MCF51XX_SCI_INIT_STRUCT _bsp_sci1_init = {
   /* queue size         */ BSPCFG_SCI2_QUEUE_SIZE,
   /* Channel            */ MCF51XX_SCI0,
   /* Clock Speed        */ BSP_SYSTEM_CLOCK,
   /* SCI1C1 Value       */ 0,
   /* SCI1C2 Value       */ 0,
   /* SCI1C3 Value       */ 0,
   /* Baud rate          */ BSPCFG_SCI1_BAUD_RATE,
   /* RX Interrupt vect  */ MCF51CN_INT_Vsci2rx,
   /* TX Interrupt vect  */ MCF51CN_INT_Vsci2tx,
   /* ER Interrupt vect  */ MCF51CN_INT_Vsci2err
};

/*****************************************************************************/
/*! GPIO Input pin table                                                     */
/*****************************************************************************/
static const GPIO_PIN_STRUCT s_tGpioInterrupt[] = 
{
  GPIO_PORT_TE | GPIO_PIN5 | GPIO_PIN_IRQ_RISING,
  GPIO_LIST_END
};

/*****************************************************************************/
/*! GPIO chip select pin                                                     */
/*****************************************************************************/
static const GPIO_PIN_STRUCT s_tGpioChipSelect[] =
{
  GPIO_PORT_TC | GPIO_PIN4 | GPIO_PIN_STATUS_1,
  GPIO_LIST_END
};

/*****************************************************************************/
/*! Init FlexBus                                                             */
/*****************************************************************************/
static void init_flexbus() 
{
  /* J0 - J5 */
  s_ptMCF51CN->GPIO.PTJPF |= MCF51XX_PTJPF_MB_MUX_SIGNALS;

  /* G0 - G7 */ 
  s_ptMCF51CN->GPIO.PTGPF |= MCF51XX_PTGPF_MB_MUX_SIGNALS;

  /* F0 - F7 */ 
  s_ptMCF51CN->GPIO.PTFPF |= MCF51XX_PTFPF_MB_MUX_SIGNALS;

  /* H0 - H7 */ 
  s_ptMCF51CN->GPIO.PTHPF |= MCF51XX_PTHPF_MB_MUX_SIGNALS;

  /* E7 - E6 */ 
  s_ptMCF51CN->GPIO.PTEPF |= MCF51XX_PTEPF_X6_MB_FB_D0 | MCF51XX_PTEPF_X7_MB_FB_CS0;
 
  /* C4 */ 
  s_ptMCF51CN->GPIO.PTCPF |= (1<<MCF51XX_PTXPF_X4_BITNUM);

  /* Chip-Select Mask Registers */
  s_ptMCF51CN->MB.CSMR0 |= MCF5XXX_FBCS_CSMR_V;
  
  /* Chip-Select Address Register @ 0x00400000 */
  s_ptMCF51CN->MB.CSAR0 = MCF5XXX_FBCS_CSAR_BA(0x400000);
  
  /* Chip-Select Control Registers */
  #if (NX50_PARDPM8 == 1) 
    s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_PS_8; /* 8-bit port size */
    s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_MUX;  /* multiplexed configuration */
  #elif (NX50_PARDPM16 == 1)
    s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_PS_16; /* 16-bit port size */
    s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_MUX;   /* multiplexed configuration */
  #endif
  s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_AA;      /* Auto-acknowledge enable */
  s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_WS(5);   /* Wait states */
  s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_ASET(1); /* Assert FB_CSn on second rising clock 
                                                         edge after address is asserted */
  s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_WRAH(0);
  s_ptMCF51CN->MB.CSCR0 |= MCF5XXX_FBCS_CSCR_RDAH(0);
}

/*****************************************************************************/
/*! Init SPI                                                                 */
/*****************************************************************************/
static void init_spi(void)
{
  /* Set pin function for port C pins 5,6, and 7*/
  s_ptMCF51CN->GPIO.PTCPF = (0b10 << MCF51XX_PTXPF_X5_BITNUM) |
                            (0b10 << MCF51XX_PTXPF_X6_BITNUM) |
                            (0b10 << MCF51XX_PTXPF_X7_BITNUM);
 
  /* SPI Baud Rate Register */
  s_ptMCF51CN->SPI1.SPIBR  = MCF5XXX_SPI8_SPIBR_SPR(0); /* SPI Baud Rate Divisor = 2 */ 
  s_ptMCF51CN->SPI1.SPIBR |= 0x10; /* Prescaler Divisor = 2  => SPI Clk: 6.125MHz 
                                      MCF5XXX_SPI8_SPIBR_SPPR(x) is buggy             */
  
  /* SPI Control Register */
  s_ptMCF51CN->SPI1.SPIC1 = MCF5XXX_SPI8_SPIC1_SPE  | MCF5XXX_SPI8_SPIC1_MSTR |
                            MCF5XXX_SPI8_SPIC1_CPOL | MCF5XXX_SPI8_SPIC1_CPHA;
  
  s_ptMCF51CN->SPI1.SPIC2 = 0x00;

  s_pvChipSelect  = fopen("gpio:output", (char*)&s_tGpioChipSelect);
}

/*****************************************************************************/
/*! Assert chip select
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiAssert(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  ioctl(s_pvChipSelect, GPIO_IOCTL_WRITE_LOG0, (char*)&s_tGpioChipSelect);
}

/*****************************************************************************/
/*! Deassert chip select
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiDeassert(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  ioctl(s_pvChipSelect, GPIO_IOCTL_WRITE_LOG1, (char*)&s_tGpioChipSelect);
}

/*****************************************************************************/
/*! Transfer byte stream via SPI. As send and receive buffer are optional,
*   the routine must be capable of sending dummy bytes (in case pbSend == NULL)
*   and discard receive bytes (if pbRecv == NULL). If caller does not pass
*   any buffer at all, we are dealing with an idle transfer to waste some time.
*   \param pvOSDependent OS Dependent parameter
*   \param pbSend        Send buffer (Can be NULL for polling data from slave)
*   \param pbRecv        Receive buffer (Can be NULL if slaves received data
                         is discarded by caller)
*   \param ulLen         Length of SPI transfer                              */
/*****************************************************************************/
void OS_SpiTransfer(void* pvOSDependent, uint8_t* pbSend, uint8_t* pbRecv, uint32_t ulLen)
{
  UNREFERENCED_PARAMETER(pvOSDependent);

  while (ulLen--)
  {
    /* Read SPI status reg before sending byte */
    while(!(s_ptMCF51CN->SPI1.SPIS&MCF5XXX_SPI8_SPIS_SPTEF)) {};

    /* write byte to spi bus */
    if (pbSend == NULL)
      s_ptMCF51CN->SPI1.SPID = 0x00;
    else
      s_ptMCF51CN->SPI1.SPID = *(pbSend++);

    /* Read SPI status reg before reading byte */
    while(!(s_ptMCF51CN->SPI1.SPIS&MCF5XXX_SPI8_SPIS_SPRF)) {};

    /* get the received byte */
    if (pbRecv == NULL)
      s_ptMCF51CN->SPI1.SPID;
    else
      *(pbRecv++) = s_ptMCF51CN->SPI1.SPID;
  }
}

/*****************************************************************************/
/*! O/S Specific initialization (initializes libpciaccess)
*     \return CIFX_NO_ERROR on success                                       */
/*****************************************************************************/
long OS_Init(void)
{
  #if (NX50_PARDPM8 == 1) | (NX50_PARDPM16 == 1)
    /* Init flexbus */
    init_flexbus();
    MQX_ParDPM_Init();    
  #elif (NX50_SERDPM == 1)  
    /* Init SPI */
    init_spi();
  #endif
  
  return CIFX_NO_ERROR;
}

/*****************************************************************************/
/*! O/S Specific de-initialization (de-initializes libpciaccess)             */
/*****************************************************************************/
void OS_Deinit(void)
{
}

/*****************************************************************************/
/*! Memory allocation wrapper (standard malloc)
*     \param ulSize Size of block to allocate
*     \return NULL on failure                                                */
/*****************************************************************************/
void* OS_Memalloc(uint32_t ulSize)
{
  return malloc(ulSize);
}

/*****************************************************************************/
/*! Memory de-allocation wrapper (standard free)
*     \param pvMem  Block to free                                            */
/*****************************************************************************/
void OS_Memfree(void* pvMem)
{
  free(pvMem);
}

/*****************************************************************************/
/*! Memory resize wrapper (standard realloc)
*     \param pvMem      Block to resize
*     \param ulNewSize  New size of the block
*     \return NULL on error                                                  */
/*****************************************************************************/
void* OS_Memrealloc(void* pvMem, uint32_t ulNewSize)
{
  return realloc(pvMem, ulNewSize);
}

/*****************************************************************************/
/*! Memset wrapper
*     \param pvMem   Memory to set
*     \param bFill   Fill byte
*     \param ulSize  Size of the fill block                                  */
/*****************************************************************************/
void OS_Memset(void* pvMem, uint8_t bFill, uint32_t ulSize)
{
  memset(pvMem, bFill, ulSize);
}

/*****************************************************************************/
/*! Memcopy wrapper
*     \param pvDest  Destination pointer
*     \param pvSrc   Source pointer
*     \param ulSize  Size to copy                                            */
/*****************************************************************************/
void OS_Memcpy(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memcpy(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Memcompare wrapper
*     \param pvBuf1  First compare buffer
*     \param pvBuf2  Second compare buffer
*     \param ulSize  Size to compare
*     \return 0 if blocks are equal                                          */
/*****************************************************************************/
int OS_Memcmp(void* pvBuf1, void* pvBuf2, uint32_t ulSize)
{
  return memcmp(pvBuf1, pvBuf2, ulSize);
}

/*****************************************************************************/
/*! Memmove wrapper (Overlapping memory copy)
*     \param pvDest  Destination buffer
*     \param pvSrc   Source buffer
*     \param ulSize  Size to move                                            */
/*****************************************************************************/
void OS_Memmove(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memmove(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Read PCI configuration area of specified card
*     \param pvOSDependent OS Dependent parameter to identify card
*     \return Pointer to configuration data (passed to WritePCIConfig)       */
/*****************************************************************************/
void* OS_ReadPCIConfig(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  return NULL;
}

/*****************************************************************************/
/*! Restore PCI configuration
*     \param pvOSDependent OS Dependent parameter to identify card
*     \param pvPCIConfig   Pointer returned from ReadPCIConfig               */
/*****************************************************************************/
void OS_WritePCIConfig(void* pvOSDependent, void* pvPCIConfig)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  UNREFERENCED_PARAMETER(pvPCIConfig);
}

static void* s_pvIsrDevInst = NULL;
/*****************************************************************************/
/*! Interrupt Routine for DPM interrupt                                      */
/*****************************************************************************/
static void InterruptServiceRoutine(void* pvParam)
{
  PDEVICEINSTANCE ptDevInst = (PDEVICEINSTANCE)s_pvIsrDevInst;
  int iISRRet               = CIFX_TKIT_IRQ_OTHERDEVICE;

  iISRRet = cifXTKitISRHandler(ptDevInst, 1);
 
  switch(iISRRet)
  {
    case CIFX_TKIT_IRQ_DSR_REQUESTED:
      /* Signal interrupt thread to call dsr handler */
      _lwsem_post(&s_tIsrSem);
      break;
    
    case CIFX_TKIT_IRQ_HANDLED:
      /* Everything was done by ISR, no need to call DSR */
      break;
    
    case CIFX_TKIT_IRQ_OTHERDEVICE:
    default:
      /* This is an interrupt from another device. pciIntConnect creates a linked 
         list of ISRs that are sharing the same interrupt line. When the interrupt 
         occurs, it sequentially invokes all the ISRs connected to it.  */
      break;
  }
}

/*****************************************************************************/
/*! Interrupt thread for DPM interrupt                                       */
/*****************************************************************************/
void InterruptServiceThread(uint_32 ulParam)
{
  PDEVICEINSTANCE ptDevInst = (PDEVICEINSTANCE)ulParam;
  
  while (1)
  {
    if (MQX_OK != _lwsem_wait(&s_tIsrSem))
      break;
    
    cifXTKitDSRHandler(ptDevInst);
  }
  
  _task_block();
}

/*****************************************************************************/
/*! Enable interrupts on the given device
*     \param pvOSDependent Pointer to internal device structure              */
/*****************************************************************************/
void OS_EnableInterrupts(void* pvOSDependent)
{
  #define CIFXISR_TASK       3
  
  if (NULL == pvOSDependent)
    return;
  
  s_pvIsrDevInst = pvOSDependent;
  
  /* Enable Interrupt on PTE5/KBI2P5 */
  s_pvInterrupt = fopen("gpio:input", (char*)&s_tGpioInterrupt);  
  
  /* Register interrupt service routine */
  ioctl(s_pvInterrupt, GPIO_IOCTL_SET_IRQ_FUNCTION, InterruptServiceRoutine);
 
  /* Start interrupt service thread */
  if (_lwsem_create(&s_tIsrSem, 0) == MQX_OK)
  {
    _task_create(0, CIFXISR_TASK, (unsigned long)pvOSDependent);
  }

  /* Enable interrupt */
  ioctl(s_pvInterrupt, GPIO_IOCTL_ENABLE_IRQ, NULL);
  
#if 0
  s_ptMCF51CN->GPIO.PTEPF  |= 0b01 << MCF51XX_PTXPF_X5_BITNUM; /* KBI2P5 */
  s_ptMCF51CN->GPIO.KBI2SC &= ~MCF51XX_KBIXSC_KBIE; /* Mask interrupts by clearing  */
  s_ptMCF51CN->GPIO.KBI2ES |=  MCF51XX_PTXDD_PTXDD5_MASK; /* detects rising edge/high level for interrupt generation */
  s_ptMCF51CN->GPIO.PTEPE  &= ~MCF51XX_PTXDD_PTXDD5_MASK; /* Internal pull-up device disabled */
  s_ptMCF51CN->GPIO.KBI2PE |=  MCF51XX_PTXDD_PTXDD5_MASK; /* Pin enabled as interrupt */
  s_ptMCF51CN->GPIO.KBI2SC |=  MCF51XX_KBIXSC_KBIMOD; /* pins detect edges and levels */
  s_ptMCF51CN->GPIO.KBI2SC |=  MCF51XX_KBIXSC_KBACK; /* clear any false interrupts */
  s_ptMCF51CN->GPIO.KBI2SC |=  MCF51XX_KBIXSC_KBIE; /* KBI Interrupt enable */
#endif
}

/*****************************************************************************/
/*! Disable interrupts on the given device
*     \param pvOSDependent Pointer to internal device structure              */
/*****************************************************************************/
void OS_DisableInterrupts(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  
  ioctl(s_pvInterrupt, GPIO_IOCTL_DISABLE_IRQ, NULL);
  fclose(s_pvInterrupt);

  s_pvIsrDevInst = NULL;
}

/*****************************************************************************/
/*! Open file for reading
*     \param szFilename   File to open (including path)
*     \param pulFileSize  Returned size of the file in bytes
*     \return Handle to the file, NULL on failure                            */
/*****************************************************************************/
void* OS_FileOpen(char* szFile, uint32_t* pulFileLen)
{
  UNREFERENCED_PARAMETER(szFile);
  UNREFERENCED_PARAMETER(pulFileLen);
  return NULL;
}

/*****************************************************************************/
/*! Read data from file
*     \param pvFile    Handle to the file (acquired by OS_FileOpen)
*     \param ulOffset  Offset to read from
*     \param ulSize    Size to read
*     \param pvBuffer  Buffer to read data into
*     \return number of bytes read                                           */
/*****************************************************************************/
uint32_t OS_FileRead(void* pvFile, uint32_t ulOffset, uint32_t ulSize, void* pvBuffer)
{
  UNREFERENCED_PARAMETER(pvFile);
  UNREFERENCED_PARAMETER(ulOffset);  
  UNREFERENCED_PARAMETER(ulSize);
  UNREFERENCED_PARAMETER(pvBuffer);
  return 0;
}

/*****************************************************************************/
/*! Close open file
*     \param pvFile    Handle to the file (acquired by OS_FileOpen)          */
/*****************************************************************************/
void OS_FileClose(void* pvFile)
{
  UNREFERENCED_PARAMETER(pvFile);
}

/*****************************************************************************/
/*! Get Millisecond counter value (used for timeout handling)
*     \return Counter value with a resolution of 1ms                         */
/*****************************************************************************/
uint32_t OS_GetMilliSecCounter(void)
{
  TIME_STRUCT tTime;
  _time_get_elapsed(&tTime);
  return (tTime.SECONDS*1000) + tTime.MILLISECONDS;  
}

/*****************************************************************************/
/*! Sleep for the given time
*     \param ulSleepTimeMs Time in ms to sleep (0 will sleep for 50us)       */
/*****************************************************************************/
void OS_Sleep(uint32_t ulSleepTimeMs)
{
  _time_delay(ulSleepTimeMs);
}

/*****************************************************************************/
/*! Create Lock (Usually same as mutex, but does not support timed waiting)
*     \return Handle to created lock                                         */
/*****************************************************************************/
void* OS_CreateLock(void)
{
  return OS_CreateMutex();
}

/*****************************************************************************/
/*! Acquire a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_EnterLock(void* pvLock)
{
  OS_WaitMutex(pvLock, 0);
}

/*****************************************************************************/
/*! Release a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_LeaveLock(void* pvLock)
{
  OS_ReleaseMutex(pvLock);
}

/*****************************************************************************/
/*! Delete a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_DeleteLock(void* pvLock)
{
  OS_DeleteMutex(pvLock);
}

/*****************************************************************************/
/*! Create mutex
*     \return Handle to new created mutex                                    */
/*****************************************************************************/
void* OS_CreateMutex(void)
{
  MUTEX_STRUCT      *ptMutex = OS_Memalloc(sizeof(MUTEX_STRUCT));
  MUTEX_ATTR_STRUCT tAttr;
  
  /* Initialize mutex attributes: */
  if (_mutatr_init(&tAttr) != MQX_OK)
  {
     printf("Initializing mutex attributes failed.\n");
     OS_Memfree(ptMutex);
     ptMutex = NULL;
  }
  
  /* Initialize the mutex: */ 
  if (_mutex_init(ptMutex, &tAttr) != MQX_OK) 
  {
     printf("Initializing print mutex failed.\n");
     OS_Memfree(ptMutex);
     ptMutex = NULL;
  }
  
  return ptMutex; 
}

/*****************************************************************************/
/*! Try to acquire mutex with timeout
*     \param pvMutex   Handle to mutex
*     \param ulTimeout Timeout in ms to wait for mutex
*     \return !=0 if mutex was acquired                                      */
/*****************************************************************************/
int OS_WaitMutex(void* pvMutex, uint32_t ulTimeout)
{ 
  UNREFERENCED_PARAMETER(ulTimeout);
  
  if (_mutex_lock((MUTEX_STRUCT*)pvMutex) != MQX_OK) 
  {
     printf("Mutex lock failed.\n");
     return 0;
  }
  return 1;
}

/*****************************************************************************/
/*! Release previously acquired mutex
*     \param pvMutex   Handle to mutex                                       */
/*****************************************************************************/
void OS_ReleaseMutex(void* pvMutex)
{
  _mutex_unlock((MUTEX_STRUCT*)pvMutex);
}

/*****************************************************************************/
/*! Delete mutex
*     \param pvMutex   Handle to mutex                                       */
/*****************************************************************************/
void OS_DeleteMutex(void* pvMutex)
{
  _mutex_destroy((MUTEX_STRUCT*)pvMutex);
  OS_Memfree(pvMutex);
}

/*****************************************************************************/
/*! Create event
*     \return Handle to created event                                        */
/*****************************************************************************/
void* OS_CreateEvent(void)
{
  LWSEM_STRUCT* ptSem = OS_Memalloc(sizeof(LWSEM_STRUCT));
  if (_lwsem_create(ptSem, 0) != MQX_OK)
  {
    OS_Memfree(ptSem);
    ptSem = NULL;
  }
  
  return ptSem;
}

/*****************************************************************************/
/*! Signal event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_SetEvent(void* pvEvent)
{
  _lwsem_post((LWSEM_STRUCT_PTR)pvEvent);
}

/*****************************************************************************/
/*! Reset event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_ResetEvent(void* pvEvent)
{
}

/*****************************************************************************/
/*! Delete event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_DeleteEvent(void* pvEvent)
{
  if (_lwsem_destroy((LWSEM_STRUCT_PTR)pvEvent) == MQX_OK)
  {
    OS_Memfree(pvEvent);
  }
  
}

/*****************************************************************************/
/*! Wait for event
*     \param pvEvent   Handle to event
*     \param ulTimeout Timeout in ms to wait for event
*     \return CIFX_EVENT_SIGNALLED if event was set, CIFX_EVENT_TIMEOUT otherwise */
/*****************************************************************************/
uint32_t OS_WaitEvent(void* pvEvent, uint32_t ulTimeout)
{
  _mqx_uint ulTicks = ulTimeout*1000*_time_get_ticks_per_sec();
  if (_lwsem_wait_ticks((LWSEM_STRUCT_PTR)pvEvent, ulTicks) != MQX_OK)
  {
    return CIFX_EVENT_TIMEOUT;
  }
  return CIFX_EVENT_SIGNALLED;
}

/*****************************************************************************/
/*! Map driver pointer to user space
*     \param pvDriverMem  Pointer to driver memory
*     \param ulMemSize    Size of the memory to map
*     \param ppvMappedMem Returned mapped pointer
*     \param os_dependent OS Dependent parameter in DEVICEINSTANCE
*     \return Handle to mapping, NULL on error                               */
/*****************************************************************************/
void* OS_MapUserPointer(void* pvDriverMem, uint32_t ulMemSize, void** ppvMappedMem, void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(ulMemSize);
  UNREFERENCED_PARAMETER(pvOSDependent);
  /* We are running in user mode, so it is not necessary to map anything to user space */ 
  *ppvMappedMem = pvDriverMem;
  
  return pvDriverMem;
}

/*****************************************************************************/
/*! Unmap previously mapped user space pointer 
*     \param phMapping  Handle returned from OS_MapUserPointer
*     \param os_dependent OS Dependent parameter in DEVICEINSTANCE
*     \return 0 on error                                                     */
/*****************************************************************************/
int OS_UnmapUserPointer(void* phMapping, void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(phMapping);
  UNREFERENCED_PARAMETER(pvOSDependent);
  /* We are running in user mode, so it is not necessary to map anything to user space */ 
  return 1;
}

/*****************************************************************************/
/*! Compare strings
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strcmp(const char* pszBuf1, const char* pszBuf2)
{
  return strcmp(pszBuf1, pszBuf2);
}

/*****************************************************************************/
/*! Compare strings case insensitive
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \param ulLen    Maximum length to compare
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strnicmp(const char* pszBuf1, const char* pszBuf2, uint32_t ulLen)
{
  char* szBuff1Copy = malloc(strlen(pszBuf1) + 1);
  char* szBuff2Copy = malloc(strlen(pszBuf2) + 1);
  int   iRet;
  int   iTmp;

  for(iTmp = 0; iTmp < min(strlen(pszBuf1), ulLen); ++iTmp)
    szBuff1Copy[iTmp] = (char)tolower((int)pszBuf1[iTmp]);

  for(iTmp = 0; iTmp < min(strlen(pszBuf2), ulLen); ++iTmp)
    szBuff2Copy[iTmp] = (char)tolower((int)pszBuf2[iTmp]);

  iRet = strncmp(szBuff1Copy, szBuff2Copy, ulLen);

  free(szBuff1Copy);
  free(szBuff2Copy);

  return iRet;
}

/*****************************************************************************/
/*! Get length of string
*     \param szText  Text buffer
*     \return Length of given string                                         */
/*****************************************************************************/
int OS_Strlen(const char* szText)
{
  return (int)strlen(szText);
}

/*****************************************************************************/
/*! Copy string to destination buffer
*     \param szText   Destination string
*     \param szSource Source string
*     \param ulLen    Maximum length to copy
*     \return Pointer to szDest                                              */
/*****************************************************************************/
char* OS_Strncpy(char* szDest, const char* szSource, uint32_t ulLen)
{
  return strncpy(szDest, szSource, ulLen);
}

#ifdef CIFX_TOOLKIT_ENABLE_DSR_LOCK
/*****************************************************************************/
/*! This functions needs to provide a lock against the interrupt service 
*   routine of the device. The easiest way is an IRQ lock but some operating 
*   systems provide a way to lock against a specific interrupt
*     \param pvODDependent OS Dependent parameter in DEVICEINSTANCE          */
/*****************************************************************************/
void OS_IrqLock(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  _int_disable();
}

/*****************************************************************************/
/*! This function re-enables the device's interrupt service routine.
*     \param pvODDependent OS Dependent parameter in DEVICEINSTANCE          */
/*****************************************************************************/
void OS_IrqUnlock(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  _int_enable();
}
#endif
